/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id:  $:

  Description:
    Implementation of the "Device/Open" dialog


  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2020-02-06  End enumeration loop if CIFX driver is selected (not a remote driver)
                and an error is returned by xDriverEnumBoards()
                signals an error xDriverEnumBoards() function
    2020-01-27  changed error evaluation in xDriverEnumBoards() function
    2006-06-28  initial version

**************************************************************************************/

///////////////////////////////////////////////////////////////////////////////////////////
/// \file CifxdeviceOpenDlg.cpp
///  Implementation of the ""Device/Open" dialog
///////////////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "cifXTest.h"
#include "CifxDeviceOpenDlg.h"
#include "cifxuser.h"
#include "cifxerrors.h"
#include "CifxDeviceSys.h"
#include "CifXDeviceChannel.h"
#include ".\cifxdeviceopendlg.h"

IMPLEMENT_DYNAMIC(CCifxDeviceOpenDlg, CDialog)

/////////////////////////////////////////////////////////////////////////////
/// Default Constructor
///   \param pParent Parent Window
/////////////////////////////////////////////////////////////////////////////
CCifxDeviceOpenDlg::CCifxDeviceOpenDlg(CWnd* pParent /*=NULL*/)
  : CDialog(CCifxDeviceOpenDlg::IDD, pParent)
  , m_pcDevice(NULL)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CCifxDeviceOpenDlg::~CCifxDeviceOpenDlg()
{
}

/////////////////////////////////////////////////////////////////////////////
/// DDX/DDV support
///   \param pDX
/////////////////////////////////////////////////////////////////////////////
void CCifxDeviceOpenDlg::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  DDX_Control(pDX, IDC_TREE_DEVICES, m_cDeviceTreeCtrl);
  DDX_Control(pDX, IDC_PROPERTY_LIST, m_cPropertyListCtrl);
}


BEGIN_MESSAGE_MAP(CCifxDeviceOpenDlg, CDialog)
  ON_WM_DESTROY()
  ON_NOTIFY(TVN_DELETEITEM, IDC_TREE_DEVICES, OnTvnDeleteitemTreeDevices)
  ON_NOTIFY(TVN_SELCHANGED, IDC_TREE_DEVICES, OnTvnSelchangedTreeDevices)
  ON_BN_CLICKED(IDOK, OnBnClickedOk)
  ON_NOTIFY(NM_DBLCLK, IDC_TREE_DEVICES, OnNMDblclkTreeDevices)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
/// First time dialog initialization. Setup columns in property view and fillup tree
///   \return TRUE
/////////////////////////////////////////////////////////////////////////////
BOOL CCifxDeviceOpenDlg::OnInitDialog()
{
  CDialog::OnInitDialog();

  CRect rectWnd;
  m_cPropertyListCtrl.GetWindowRect(rectWnd);

  int iWidth = rectWnd.Width() - GetSystemMetrics(SM_CXVSCROLL) - 2 * GetSystemMetrics(SM_CXEDGE);

  m_cPropertyListCtrl.InsertColumn(0, _T("Property"), LVCFMT_LEFT, iWidth / 2);
  m_cPropertyListCtrl.InsertColumn(1, _T("Value"),    LVCFMT_LEFT, iWidth / 2);

  m_cPropertyListCtrl.SetExtendedStyle(LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES);

  // Enumerate the boards
  for (std::vector<CCifXAPIlib*>::iterator iterDrv = m_pcvCifXAPI.begin(); iterDrv != m_pcvCifXAPI.end(); iterDrv++)
  {
    if (!(*iterDrv)->Valid())
      continue;

    BOARD_INFORMATION tBoardInfo = {0};
    unsigned long     ulBoardIdx = 0;
    long lBoardRet = CIFX_NO_ERROR;
    while(lBoardRet == CIFX_NO_ERROR)
    {
      lBoardRet = (*iterDrv)->DriverEnumBoards(ulBoardIdx, sizeof(tBoardInfo), &tBoardInfo);

      if(lBoardRet == CIFX_NO_MORE_ENTRIES)
      {
        // No more boards
        break;
      } else if (lBoardRet != CIFX_NO_ERROR)
      {
        /* Attention, only go on with the enumeration until CIFX_NO_MORE_ENTRIES if it is NOT the CIFX driver.
           Because this driver currently returns CIFX_INVALID_HANDLE and not the expected CIFX_NO_MORE_ENTRIES,
           results in an endless loop! */
        if ((*iterDrv)->IsRemoteAPI())
          lBoardRet = CIFX_NO_ERROR;
      }else
      {
        CCifXDeviceBase* pcDevice = new CCifXDeviceSys(&tBoardInfo, NULL, (*iterDrv));

        HTREEITEM hDeviceItem = m_cDeviceTreeCtrl.InsertItem(pcDevice->GetDeviceName());
        m_cDeviceTreeCtrl.SetItemData(hDeviceItem, (DWORD_PTR)pcDevice);

        // Board found
        unsigned long ulChannelIdx  = 0;
        long          lChannelRet   = CIFX_NO_ERROR;

        //------------------------------------------
        // Find out how many channles are active
        // on this board
        //------------------------------------------
        while(lChannelRet == CIFX_NO_ERROR)
        {
          // Read all channel information from the given board
          CHANNEL_INFORMATION tChannelInfo = {0};
          lChannelRet = (*iterDrv)->DriverEnumChannels(ulBoardIdx, ulChannelIdx, sizeof(tChannelInfo), &tChannelInfo);
          if(lChannelRet == CIFX_NO_ERROR)
          {
            CCifXDeviceChannel* pcDevice = new CCifXDeviceChannel(ulChannelIdx, &tBoardInfo, &tChannelInfo, (*iterDrv));

            HTREEITEM hChannelItem = m_cDeviceTreeCtrl.InsertItem(pcDevice->GetShortDeviceName(), 0, 0, hDeviceItem);
            m_cDeviceTreeCtrl.SetItemData(hChannelItem, (DWORD_PTR)pcDevice);
          }
          ++ulChannelIdx;    // Next channel
        }

        m_cDeviceTreeCtrl.Expand(hDeviceItem, TVE_EXPAND);

      }
      ++ulBoardIdx;    // Next board
    }
  }

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}

/////////////////////////////////////////////////////////////////////////////
/// Dialog WM_DESTROY message handler
/////////////////////////////////////////////////////////////////////////////
void CCifxDeviceOpenDlg::OnDestroy()
{
}

/////////////////////////////////////////////////////////////////////////////
/// Event when a tree item is being removed from the tree
///   \param *pNMHDR
///   \param *pResult
/////////////////////////////////////////////////////////////////////////////
void CCifxDeviceOpenDlg::OnTvnDeleteitemTreeDevices(NMHDR *pNMHDR, LRESULT *pResult)
{
  LPNMTREEVIEW pNMTreeView = reinterpret_cast<LPNMTREEVIEW>(pNMHDR);

  CCifXDeviceBase* pcDevice = (CCifXDeviceBase*)m_cDeviceTreeCtrl.GetItemData(pNMTreeView->itemOld.hItem);
  delete pcDevice;

  *pResult = 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Event when a tree item selection changes
///   \param *pNMHDR
///   \param *pResult
/////////////////////////////////////////////////////////////////////////////
void CCifxDeviceOpenDlg::OnTvnSelchangedTreeDevices(NMHDR *pNMHDR, LRESULT *pResult)
{
  LPNMTREEVIEW pNMTreeView = reinterpret_cast<LPNMTREEVIEW>(pNMHDR);

  CCifXDeviceBase* pcDevice = (CCifXDeviceBase*)m_cDeviceTreeCtrl.GetItemData(pNMTreeView->itemNew.hItem);

  m_cPropertyListCtrl.DeleteAllItems();

  std::vector<CCifXDeviceBase::PROPERTY_LIST> cvList = pcDevice->GetDescription();

  for(UINT uiIdx = 0; uiIdx < cvList.size(); ++uiIdx)
  {
    m_cPropertyListCtrl.InsertItem(uiIdx, cvList[uiIdx].csProperty);
    m_cPropertyListCtrl.SetItemText(uiIdx, 1, cvList[uiIdx].csValue);
  }

  *pResult = 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Get the user selected device
///   \return Device the user had selected when pressing OK
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceBase* CCifxDeviceOpenDlg::GetSelectedDevice(void)
{
  return m_pcDevice;
}

/////////////////////////////////////////////////////////////////////////////
/// OK Button clicked handler
/////////////////////////////////////////////////////////////////////////////
void CCifxDeviceOpenDlg::OnBnClickedOk()
{
  HTREEITEM hSelection = m_cDeviceTreeCtrl.GetSelectedItem();

  if(NULL != hSelection)
  {
    m_pcDevice = (CCifXDeviceBase*)m_cDeviceTreeCtrl.GetItemData(hSelection);

    //remove data, to prevent deletion on dialog closing
    m_cDeviceTreeCtrl.SetItemData(hSelection, NULL);
  }

  OnOK();
}

/////////////////////////////////////////////////////////////////////////////
/// Device Tree Double-clicked event
/////////////////////////////////////////////////////////////////////////////
void CCifxDeviceOpenDlg::OnNMDblclkTreeDevices(NMHDR* /*pNMHDR*/, LRESULT* pResult)
{
  HTREEITEM hItem = m_cDeviceTreeCtrl.GetNextItem(TVI_ROOT, TVGN_CARET);

  if(NULL != hItem)
  {
    OnBnClickedOk();
  }
  *pResult = 0;
}
